{
==============================================================================
Copyright (C) 2006  anir (anir<AT>k.ro)

Modified 2010 - Mark Cranness:
- Work on Windows 7 when pointer speed < 6/11 or 'Enhance pointer precision' is set.
- Better matching of mouse movements to cursor movements.
- MUCH lower CPU use.
- Compiles with freepascal 2.4 http://www.freepascal.org/
	Command line to compile: fpc -Sd MouseMovementRecorder

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
==============================================================================
}

{$apptype console}
{$r MouseMovementRecorder.rc}  // Avoid 'DPI Virtualisation' + Version info

uses
	Windows,
	DirectInput; // http://www.clootie.ru/delphi/DX92/Clootie_DirectX92.exe
	// Unrar and copy DirectInput.pas & DXTypes.pas from the Borland_D6-7\ folder
	// to same folder as MouseMovementRecorder.pas

const
	Caption0 = 'MOUSE MOVEMENT    POINTER MOVEMENT  FREQUENCY ';
	Caption1 = ' EnPtPr ' + #13;
	ErrMsgPos = 'ERROR: GetCursorPos';
	ErrMsgPrfC = 'ERROR: QueryPerformanceCounter';
	ErrMsgPrfF = 'ERROR: QueryPerformanceFrequency';
	ErrMsgDIState = 'ERROR: DirectInput.GetDeviceState';
	FOREGROUND_WHITE = FOREGROUND_BLUE or FOREGROUND_GREEN or FOREGROUND_RED;

type
	MouseAccel = record
		MouseThreshold1 : integer;
		MouseThreshold2 : integer;
		MouseSpeed : integer;
	end;

var
	DIMouDev: IDirectInputDevice8;
	DIMouDat, DIMouDat2: DIMouseState;
	hMouseEvent: HANDLE;
	ConHdl, ErrHdl, TmpLwd: longword;
	OldPos, CurPos: TPoint;
	CursorX, CursorY: integer;
	PrfFrq, OldCnt, PrfCnt: Int64;
	PtrCatchup : Int64;
	MouseFrq: integer;
	Accel: MouseAccel;
	SeenEPP: boolean;
	TryToCatchup, TryToCatchupEnabled: boolean;
	i: integer;
	Not1To1, Not1To1Count, Is1To1Count, PrevNot1To1: integer;

label
	Loop;

// Int2Str from http://kolmck.net/ : avoid linking all of SysUtils
function IntToStr(Value: Integer): AnsiString;
var
	Buf: Array[0..15] of AnsiChar;
	Dst: PAnsiChar;
	Minus: Boolean;
	D: DWORD;
begin
	Dst := @Buf[15];
	Dst^ := #0;
	Minus := False;
	if Value < 0 then begin
		Value := -Value;
		Minus := True;
	end;
	D := Value;
	repeat
		Dec(Dst);
		Dst^ := AnsiChar((D mod 10) + Byte('0'));
		D := D div 10;
	until D = 0;
	if Minus then begin
		Dec(Dst);
		Dst^ := '-';
	end;
	Result := Dst;
end;

procedure ErrorHalt(ErrMsg: string);
begin
	SetConsoleTextAttribute(ConHdl, FOREGROUND_RED or FOREGROUND_INTENSITY);
	WriteFile(ConHdl, ErrMsg[1], Length(ErrMsg), TmpLwd, Nil);
	SetConsoleTextAttribute(ConHdl, FOREGROUND_WHITE);
	Halt(1);
end;

function InitDI: Boolean;
var
	DInput: IDirectInput8;
begin
	hMouseEvent := CreateEvent(Nil, false, false, Nil);
	Result := (DirectInput8Create(hInstance, DIRECTINPUT_VERSION, IID_IDirectInput8, DInput, Nil) = DI_OK)
		  and (DInput.CreateDevice(GUID_SysMouse, DIMouDev, Nil) = DI_OK)
		  and (DIMouDev.SetDataFormat(c_dfDIMouse) = DI_OK)
		  and (DIMouDev.SetEventNotification(hMouseEvent) = DI_OK);
end;

begin

	// Normal works best...
	//SetPriorityClass(GetCurrentProcess, HIGH_PRIORITY_CLASS);

	// Output goes to the console (text)
	SetConsoleTitle('Mouse Movement Recorder');
	ConHdl := GetStdHandle(STD_OUTPUT_HANDLE);
	if ConHdl = INVALID_HANDLE_VALUE then ErrorHalt('ERROR: GetStdHandle');
	// Caption goes to stderr, so it does not end up in a file if we '>' redirect the output
	ErrHdl := GetStdHandle(STD_ERROR_HANDLE);
	if ErrHdl = INVALID_HANDLE_VALUE then ErrorHalt('ERROR: GetErrHandle');
	SetConsoleMode(ConHdl, ENABLE_PROCESSED_OUTPUT);
	SetConsoleMode(ErrHdl, ENABLE_PROCESSED_OUTPUT);

	// Initialise
	if not InitDI then ErrorHalt('ERROR: DirectInputCreate');
	if not QueryPerformanceFrequency(PrfFrq) then ErrorHalt(ErrMsgPrfF);
	if not QueryPerformanceCounter(OldCnt) then ErrorHalt(ErrMsgPrfC);
	if DIMouDev.Acquire <> DI_OK then ErrorHalt('ERROR: DirectInput.Acquire');
	if not GetCursorPos(OldPos) then ErrorHalt(ErrMsgPos);
	SetConsoleTextAttribute(ConHdl, FOREGROUND_WHITE);
	WriteFile(ConHdl, (Caption0 + Caption1 + #10)[1], Length(Caption0)+Length(Caption1)+1, TmpLwd, Nil);
	SeenEPP := false;
	TryToCatchupEnabled := true;
	for i := 1 to ParamCount do if LowerCase(ParamStr(i)) = '-nocatchup' then TryToCatchupEnabled := false;
	TryToCatchup := TryToCatchupEnabled; Not1To1Count := 0; Is1To1Count := 0; PrevNot1To1 := 0;

	Loop:

	// Get raw mouse data
	WaitForSingleObject(hMouseEvent, INFINITE);
	if DIMouDev.GetDeviceState(SizeOf(DIMouDat), @DIMouDat) <> DI_OK then ErrorHalt(ErrMsgDIState);
	if not QueryPerformanceCounter(PrfCnt) then ErrorHalt(ErrMsgPrfC);
	
	// Now wait (at most 0.5ms) for the pointer to catchup
	repeat
		// Get accelerated cursor (pointer) movement
		if not GetCursorPos(CurPos) then ErrorHalt(ErrMsgPos);
		// Calculate pointer movement based on old position
		CursorX := CurPos.X - OldPos.X;
		CursorY := CurPos.Y - OldPos.Y;
		if not TryToCatchupEnabled or not TryToCatchup 
				or (DIMouDat.lX = CursorX) and (DIMouDat.lY = CursorY) then
			// Pointer position has caught up (or we are not trying to catchup)
			break;
		if not QueryPerformanceCounter(PtrCatchup) then ErrorHalt(ErrMsgPrfC);
		if (PtrCatchup-PrfCnt)*2000 > PrfFrq then
			// > 0.5ms
			break;
		// Don't be greedy during a busy wait
		Sleep(0);
		//  Has there been some more mouse movement (SteelSeries Xai) ?
		if DIMouDev.GetDeviceState(SizeOf(DIMouDat2), @DIMouDat2) <> DI_OK then ErrorHalt(ErrMsgDIState);
		ResetEvent(hMouseEvent);
		DIMouDat.lX += DIMouDat2.lX;
		DIMouDat.lY += DIMouDat2.lY;
	until False;
	
	OldPos := CurPos;

	// Display mouse movement
	WriteFile(ConHdl, (IntToStr(DIMouDat.lX) + ' x ' + IntToStr(DIMouDat.lY)
			  + '             ')[1], 18, TmpLwd, Nil);

	// Display differences between mouse and pointer movement with red/green
	if (DIMouDat.lX <> CursorX) or (DIMouDat.lY <> CursorY) then begin
		if CursorX*CursorX + CursorY*CursorY > DIMouDat.lX*DIMouDat.lX + DIMouDat.lY*DIMouDat.lY then begin
			Not1To1 := +1;
			// Pointer > Mouse has a RED visual cue
			SetConsoleTextAttribute(ConHdl, FOREGROUND_WHITE or BACKGROUND_RED);
		end else begin
			Not1To1 := -1;
			// Pointer < Mouse has a GREEN visual cue
			SetConsoleTextAttribute(ConHdl, FOREGROUND_WHITE or BACKGROUND_GREEN);
		end;
		if (Not1To1 > 0) <> (Not1To1Count > 0) then Not1To1Count := 0;
		Not1To1Count += Not1To1;
		if abs(Not1To1Count) >= 3 then
			// 3 or more green in a row, or 3 or more red in a row (ignoring 1-to-1 lines):
			// STOP trying to catchup because it wastes CPU cycles for no reason
			TryToCatchup := false;
	end else begin
		Not1To1 := 0;
		if PrevNot1To1 <> 0 then Is1To1Count := 0;
		Is1To1Count += 1;
		if Is1To1Count >= 3 then begin
			// START trying to catchup because response is now 1-to-1
			TryToCatchup := true and TryToCatchupEnabled;
			Not1To1Count := 0;
		end;
	end;
	PrevNot1To1 := Not1To1;
	
	// Display pointer movement
	WriteFile(ConHdl, (IntToStr(CursorX) + ' x ' + IntToStr(CursorY)
			  + '            ')[1], 17, TmpLwd, Nil);
	SetConsoleTextAttribute(ConHdl, FOREGROUND_WHITE);

	// Display estimated mouse bus update frequency
	MouseFrq := Round(PrfFrq / (PrfCnt - OldCnt));
	OldCnt := PrfCnt;
	WriteFile(ConHdl, (' ~ ' + IntToStr(MouseFrq) + ' Hz    ')[1], 11, TmpLwd, Nil);

	// Display Accel = MouseSpeed flag
	SystemParametersInfo(SPI_GETMOUSE, 0, @Accel, 0);
	if Accel.MouseSpeed > 0 then begin
		SeenEPP := true;
		SetConsoleTextAttribute(ConHdl, FOREGROUND_WHITE or BACKGROUND_RED);
		WriteFile(ConHdl, (' ON ')[1], 4, TmpLwd, Nil);
		if Accel.MouseSpeed > 1 then
			WriteFile(ConHdl, (' (' + IntToStr(Accel.MouseSpeed) + ') ')[1], 5, TmpLwd, Nil);
		SetConsoleTextAttribute(ConHdl, FOREGROUND_WHITE);
	end else begin
		WriteFile(ConHdl, (' Off ')[1], 5, TmpLwd, Nil);
	end;
	WriteFile(ConHdl, ('     ' + #13#10)[1], 7, TmpLwd, Nil);

	// Display legend/caption at bottom
	WriteConsoleA(ErrHdl, @Caption0[1], Length(Caption0), TmpLwd, Nil);
	if SeenEPP then
		SetConsoleTextAttribute(ErrHdl,FOREGROUND_WHITE or BACKGROUND_RED);
	WriteConsoleA(ErrHdl, @Caption1[1], Length(Caption1), TmpLwd, Nil);
	SetConsoleTextAttribute(ErrHdl, FOREGROUND_WHITE);

	// Rinse and repeat (use Ctrl+C or Ctrl+Break to stop program)
	goto Loop;

end.